// Copyright 2014 Google Inc. All Rights Reserved.

#ifndef ANDROID_AUTO_PROJECTION_PROTOCOL_IAUDIO_SOURCE_CALLBACKS_H
#define ANDROID_AUTO_PROJECTION_PROTOCOL_IAUDIO_SOURCE_CALLBACKS_H

#include "common.h"

/**
 * This class represents an interface that every sink implementation must subclass if they
 * wish to implement an audio source to send over microphone data. You should look at the
 * documentation for class AudioSource for a pseudocode example of how to set up an
 * AudioSource.
 */
class IAudioSourceCallbacks {
public:
    virtual ~IAudioSourceCallbacks() { }
    /**
     * Called when an ack comes through. The ack mechanism is used for flow control and the
     * sender must maintain a counter of outstanding unacknowledged frames. If the number
     * of unacknowledged frames exceeds the maximum value set for the connection, no more
     * frames should be sent.
     * @param sessionId The id of the session to which that the frame being acknowledged
     *        belongs. You must only decrement the number of outstanding frames if the
     *        session id here is the same as the current session id.
     * @param ack The number of frames being acknowledged. The number of outstanding frames
     *        should be decremented by this value. This allows for more than one frame to
     *        be acknowledged at a time.
     */
    virtual void ackCallback(int sessionId, uint32_t ack) = 0;
    /**
     * Called when a microphone request comes through.
     * @param open True if the mic should be opened, false if it should be closed.
     * @param ancEnabled True if active noise cancellation should be enabled, false otherwise.
     * @param ecEnabled True if echo cancellation should be enabled, false otherwise.
     * @param maxUnacked The maximum number of frames that should be kept outstanding.
     * return STATUS_SUCCESS or a positive number on success, which is used as a session id,
     *        a negative error code otherwise.
     */
    virtual int microphoneRequestCallback(bool open, bool ancEnabled,
            bool ecEnabled, int maxUnacked) = 0;
};

#endif // ANDROID_AUTO_PROJECTION_PROTOCOL_IAUDIO_SOURCE_CALLBACKS_H
